import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";

type TaskAnalysisData = {
  taskStats: {
    totalTasks: number;
    completedTasks: number;
    pendingTasks: number;
    overdueTasks: number;
    recurringTasks: number;
    nonRecurringTasks: number;
    periodCompletionRate: number;
    totalPeriods: number;
    completedPeriods: number;
    tasksCreatedToday: number;
    tasksCreatedThisWeek: number;
    tasksCreatedThisMonth: number;
  };
  patternStats: Array<{
    pattern: string;
    total: number;
    completed: number;
    completionRate: number;
  }>;
  departmentStats: Array<{
    department: string;
    total: number;
    completed: number;
    overdue: number;
    nonConformities: number;
    completionRate: number;
  }>;
  technicianStats: Array<{
    technicianId: string;
    technicianName: string;
    technicianEmail: string;
    isExternal: boolean;
    companyName: string | null;
    total: number;
    completed: number;
    overdue: number;
    avgCompletionTime: number;
    completionRate: number;
  }>;
  nonConformityStats: {
    totalNonConformities: number;
    tasksWithNonConformities: number;
    nonConformityRate: number;
  };
};

interface TaskNonConformityTabProps {
  taskData: TaskAnalysisData;
}

export default function TaskNonConformityTab({
  taskData,
}: TaskNonConformityTabProps) {
  const { t } = useLanguage();
  const { nonConformityStats, departmentStats } = taskData;

  // Sort departments by non-conformities (highest first)
  const sortedByNonConformities = [...departmentStats].sort(
    (a, b) => b.nonConformities - a.nonConformities
  );

  return (
    <>
      {/* Non-Conformity Overview */}
      <div className='grid grid-cols-1 md:grid-cols-3 gap-6 mb-8'>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.totalNonConformities") || "Total Non-Conformities"}
          </h3>
          <p className='text-3xl font-bold text-red-600'>
            {nonConformityStats.totalNonConformities}
          </p>
          <p className='text-xs text-gray-500 mt-1'>{t("analysis.taskAnalysis.notOkItemsFound") || "NOT_OK items found"}</p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.tasksWithIssues") || "Tasks with Issues"}
          </h3>
          <p className='text-3xl font-bold text-orange-600'>
            {nonConformityStats.tasksWithNonConformities}
          </p>
          <p className='text-xs text-gray-500 mt-1'>
            {t("analysis.taskAnalysis.outOf") || "out of"} {taskData.taskStats.totalTasks} {t("analysis.taskAnalysis.totalTasksLabel2") || "total tasks"}
          </p>
        </div>
        <div className='bg-white p-6 rounded-lg shadow'>
          <h3 className='text-sm font-medium text-gray-500 mb-2'>
            {t("analysis.taskAnalysis.nonConformityRate") || "Non-Conformity Rate"}
          </h3>
          <p className='text-3xl font-bold text-red-600'>
            {nonConformityStats.nonConformityRate.toFixed(1)}%
          </p>
          <p className='text-xs text-gray-500 mt-1'>{t("analysis.taskAnalysis.tasksWithIssuesLabel") || "tasks with issues"}</p>
        </div>
      </div>

      {/* Non-Conformities by Department */}
      <div className='bg-white p-6 rounded-lg shadow'>
        <h3 className='text-lg font-semibold mb-4'>
          {t("analysis.taskAnalysis.nonConformitiesByDepartment") || "Non-Conformities by Department"}
        </h3>
        {sortedByNonConformities.length > 0 ? (
          <div className='overflow-x-auto'>
            <table className='min-w-full divide-y divide-gray-200'>
              <thead className='bg-gray-50'>
                <tr>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.department") || "Department"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.totalTasksLabel3") || "Total Tasks"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.nonConformities") || "Non-Conformities"}
                  </th>
                  <th className='px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.taskAnalysis.completionRateLabel") || "Completion Rate"}
                  </th>
                </tr>
              </thead>
              <tbody className='bg-white divide-y divide-gray-200'>
                {sortedByNonConformities.map((dept) => (
                  <tr key={dept.department}>
                    <td className='px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900'>
                      {dept.department}
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm text-gray-500'>
                      {dept.total}
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm'>
                      <span
                        className={`font-bold ${
                          dept.nonConformities === 0
                            ? "text-green-600"
                            : dept.nonConformities <= 2
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {dept.nonConformities}
                      </span>
                    </td>
                    <td className='px-6 py-4 whitespace-nowrap text-sm'>
                      <span
                        className={`font-medium ${
                          dept.completionRate >= 80
                            ? "text-green-600"
                            : dept.completionRate >= 50
                            ? "text-orange-600"
                            : "text-red-600"
                        }`}
                      >
                        {dept.completionRate.toFixed(1)}%
                      </span>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        ) : (
          <p className='text-gray-500 text-center py-8'>{t("analysis.taskAnalysis.noDataAvailable") || "No data available"}</p>
        )}
      </div>

      {/* Summary */}
      <div className='mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6'>
        <h3 className='text-lg font-semibold text-blue-900 mb-2'>
          {t("analysis.taskAnalysis.nonConformitySummary") || "Non-Conformity Summary"}
        </h3>
        <p className='text-sm text-blue-800'>
          {t("analysis.taskAnalysis.nonConformityDescription") || "Non-conformities are items marked as \"NOT_OK\" during task completion. These indicate issues that require attention and may result in ticket creation for maintenance or repair."}
        </p>
        <div className='mt-4 grid grid-cols-1 md:grid-cols-2 gap-4'>
          <div>
            <p className='text-xs font-medium text-blue-700 mb-1'>
              {t("analysis.taskAnalysis.averagePerTask") || "Average per Task:"}
            </p>
            <p className='text-lg font-bold text-blue-900'>
              {taskData.taskStats.totalTasks > 0
                ? (
                    nonConformityStats.totalNonConformities /
                    taskData.taskStats.totalTasks
                  ).toFixed(2)
                : "0.00"}{" "}
              {t("analysis.taskAnalysis.issues") || "issues"}
            </p>
          </div>
          <div>
            <p className='text-xs font-medium text-blue-700 mb-1'>
              {t("analysis.taskAnalysis.healthStatus") || "Health Status:"}
            </p>
            <p className='text-lg font-bold text-blue-900'>
              {nonConformityStats.nonConformityRate < 10
                ? t("analysis.taskAnalysis.excellent") || "✅ Excellent"
                : nonConformityStats.nonConformityRate < 25
                ? t("analysis.taskAnalysis.good") || "⚠️ Good"
                : t("analysis.taskAnalysis.needsAttention") || "🔴 Needs Attention"}
            </p>
          </div>
        </div>
      </div>
    </>
  );
}

